/*
 * Created on 15.01.2004 11:31:34
 *
 * Multimediale Algorithmen und Datenstrukturen Assessments
 */
package mauda;

import mauda.operation.*;
/**
 * A semantic event which indicates that a exercise-defined event
 * occured. This high-level event is generated at different places
 * in the whole project, and is normally passed through the queue
 * of <code>JedasPanel</code>, and then transmitted to every
 * <code>ExerciseUpdateListener</code> object, that registered to 
 * receive such events using the <code>Exercise</code>'s 
 * <code>addExerciseUpdateListener</code> method.
 * This event is essential for sychronizing objects with the 
 * scheduler (animation).
 * 
 * @author Markus Krebs
 */
public class ExerciseUpdateEvent {
	
	// fr spezielle Nachrichten, bei denen weitere Informationen
	// in der Message stehen
	/**
	 * ID for special events. Further event-specific information is
	 * stored in <code>message</code>
	 */
	public static final int SPECIAL = -1;
	
	/**
	 * ID that indicates that a operation was executed.
	 */
	public static final int OPERATION_EXECUTED = 0;
	/**
	 * ID that indicates that the filename of the current exercise
	 * was changed.
	 */
	public static final int FILENAME_CHANGED = 1;
	/**
	 * ID that indicates that all data-structures should delete
	 * their previous stored information related to e.g. already 
	 * executed operations. For example for
	 * <code>OperationRecorder</code> this event will clear all
	 * previous stored operations.
	 */
	public static final int CLEAR_PRECEDING = 2;
	/**
	 * ID that indicates that all data-structures should delete
	 * related to the actual position all following operations.
	 * @see mauda.ExerciseUpdateEvent#CLEAR_PRECEDING
	 */
	public static final int CLEAR_FOLLOWING = 3;
	/**
	 * ID that indicates that the exercise was finished in loading.
	 * Remember that this is normally related to the init-operations
	 * of an exercise, what means that a exercise must not have
	 * really finished loading, when the this event occures.
	 */
	public static final int FINISHED_LOADING = 4;
	/**
	 * ID that indicates to hide the animation-object
	 */
	public static final int DSOBJECT_HIDE = 5;
	/**
	 * ID that indicates to show a previously hidden animation-
	 * object.
	 */
	public static final int DSOBJECT_SHOW = 6;
	
	/**
	 * ID that indicates that an undo was made
	 */
	public static final int BACK = 7;
	/**
	 * ID that indicates that a redo was made
	 */
	public static final int FORWARD = 8;
	/**
	 * ID that indicates that a animated redo was made. This event
	 * is only fired, when <code>AnimatedUndoRedo</code> was used.
	 * @see mauda.undoredo.AnimatedUndoRedo
	 */
	public static final int FORWARD_ANIMATED = 9;
	/**
	 * ID that indicates that the exercise was cleared. That means
	 * e.g. for <code>OperationRecorder</code> to delete all stored
	 * operations.
	 */
	public static final int RESET = 10;
	
	/**
	 * ID that indicates that the plug in was changed.
	 */
	public static final int PLUGIN_CHANGED = 11;
	
	/**
	 * ID that indicates that the user is jumped to another
	 * operation. The delta-value can accessed by
	 * <code>getValue()</code>
	 * @see mauda.ExerciseUpdateEvent#getValue()
	 */
	public static final int JUMP = 12;
	
	/**
	 * ID that indicates that all operations in the current queue
	 * are fully executed by the <code>JedasPanel</code>.
	 */
	public static final int END_OF_EXECUTION = 13;
	
	/**
	 * ID that indicates that the <code>JedasPanel</code> starts
	 * the execution of passed operations/events to the queue.
	 */
	public static final int START_OF_EXECUTION = 14;
	
	/**
	 * ID that indicates that the Jedas-Animation goes in suspend-Mode.
	 * This is used, when e.g. the recording is paused.
	 */
	public static final int SUSPEND_EXECUTION = 15;
	/**
	 * ID that indicates that the Jedas-Animation resumes from suspend.
	 * This is used, when e.g. a recording was paused, and now continueing.
	 */
	public static final int RESUME_EXECUTION = 16;
	
	/**
	 * ID that indicates that an exercise is fully loaded
	 */
	public static final int FULL_LOADED = 17;
	
	/**
	 * ID that indicates that a real undo occured
	 */
	public static final int UNDO = 18;
	
	/**
	 * ID that indicates that a real redo occured
	 */
	public static final int REDO = 19;
		
	private int id;
	
	// "GO_IN_TODo-MODE" "FEEDBACK_CONFIG"
	private String message; 
	
	private int value;	// Feld fr Zahlenmessages (JUMP -> Delta)

	private SimpleExercise exercise;
	
	private SimpleOperation so;
	
	/**
	 * Creates an event
	 * 
	 * @param ex Exercise-Object
	 * @param id A ID for the event
	 */
	public ExerciseUpdateEvent(SimpleExercise ex, int id) {
		this.exercise = ex;
		this.id = id;
		this.message = null;
		this.so = null;
	}
	/**
	 * Returns the ID of the event
	 * @return ID
	 */
	public int getID() { return id; }

	
	/**
	 * Sets the message field of the event
	 * @param m
	 */
	public void setMessage(String m) { message = m; }
	/**
	 * Returns the message field of the event
	 * @return Message
	 */
	public String getMessage() { return message; }

	/**
	 * Sets the operation for the current event. This field is e.g.
	 * used in an <code>OPERATION_EXECUTED</code> event to store
	 * the executed operation.
	 * @param so
	 */
	public void setOperation(SimpleOperation so) { this.so = so; }
	/**
	 * Returns the operation.
	 * @see mauda.ExerciseUpdateEvent#setOperation(SimpleOperation)
	 * @return SimpleOperation
	 */
	public SimpleOperation getOperation() { return so; }
	
	/**
	 * Sets the value field of an event. This field is e.g. used in
	 * an <code>JUMP</code> event, to store the delta value for the
	 * distance of jumping, relativ to the actual position.
	 * @param value
	 */
	public void setValue(int value) { this.value = value;  }
	/**
	 * Returns the value-field.
	 * @return value
	 */
	public int getValue() { return value; }	
	
	/**
	 * Returns the exercise-object the event belongs to.
	 * @return Exercise
	 */
	public SimpleExercise getExercise() { return exercise; }
	
	/**
	 * Returns TRUE if the current event changes the data-structure,
	 * what means that e.g. the display have to be updated.
	 * @return true if the data-structure was changed, false otherwise
	 */
	public boolean dsChanged() {
		switch(id) {
			// DS-nderungen nur bei folgenden IDs
			case RESET :
			case OPERATION_EXECUTED :
			case BACK :
			case FORWARD :
			case FORWARD_ANIMATED :
			case UNDO :
			case REDO :
			case PLUGIN_CHANGED :
			case JUMP :
				return true;
			default :
				return false; 
		}
	}
	
	/**
	 * Returns a string-representation of the event-object
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		String s = "ID = ";
		switch(id) {
			case OPERATION_EXECUTED : s = "OPERATION_EXECUTED"; break;
			case FILENAME_CHANGED : s = "FILENAME_CHANGED"; break;
			case CLEAR_PRECEDING : s = "CLEAR_PRECEDING"; break;
			case CLEAR_FOLLOWING : s = "CLEAR_FOLLOWING"; break;
			case FINISHED_LOADING : s = "FINISHED_LOADING"; break;
			case DSOBJECT_HIDE : s = "DSOBJECT_HIDE"; break;
			case DSOBJECT_SHOW : s = "DSOBJECT_SHOW"; break;
			case BACK : s = "BACK"; break;
			case FORWARD : s = "FORWARD"; break;
			case FORWARD_ANIMATED : s = "FORWARD_ANIMATED"; break;
			case UNDO : s = "UNDO"; break;
			case REDO : s = "REDO"; break;
			case RESET : s = "RESET"; break;
			case PLUGIN_CHANGED : s = "PLUGIN_CHANGED"; break;
			case JUMP : s = "JUMP: delta = "+value; break;
			case SPECIAL : s = "SPECIAL -> Message="+message; break;
			case START_OF_EXECUTION : s = "START_OF_EXECUTION"; break;
			case END_OF_EXECUTION : s = "END_OF_EXECUTION"; break;
			case SUSPEND_EXECUTION : s = "SUSPEND_EXECUTION"; break;
			case RESUME_EXECUTION : s = "RESUME_EXECUTION"; break;
		}
		return s;
	}
	
	//public void save() {
		// id message value SimpleOperation
	//}
}
