package mauda.plugin.fibheap;
import jedas.*;
import jedas.trees.*;
import java.awt.Color;

/**
 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
 * <b>Since:</b> 0.951, version 0.99rc1<br>
 * Implementation of the <tt>LabelledNodeItem</tt> interface (which is a
 * sub-interface of <tt>NodeItem</tt>).
 * <p>A <tt>NodeObj</tt> consists of an <tt>OvalObj</tt> and three <tt>TextObj</tt>
 * instances: the node visualizer and three different labels, one to the left, one
 * in the center and one at the right of the node.</p>
 *
 * <p><tt>FibNodeObj</tt> instances do not have any other properties than these
 * labels, especially don't they have a value or a balance.</p>
 *
 * @see NodeItem
 * @see LabelledNodeItem
 * @see SearchNodeObj
 * @author Martin Danielsson (modified by Tobias Lauer, Markus Krebs)
 */
public class FibNodeObj {
	
	protected CompObj compObj;

	static final long serialVersionUID = -14585107407054923L;

	private static final double NODE_SIZE	= 0.06;
	private static final double FONT_SIZE	= 0.03;
	private static final double SPACE		= 0.01;
	private static final Color COLOR		= Color.black;
	private static final Color BACKGROUND	= Color.white;
	private static final Color HIGHLIGHT	= new Color(0xffaaaa);

	OvalObj oval;
	TextObj leftLabel, centerLabel, rightLabel;

	private DPair cRueck = new DPair();

	private transient TreeNode treeNode = null;

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Create a <tt>NodeObj</tt> instance with the given parameters.
	 * @param pos the position of the object
	 */
	public FibNodeObj( DPair pos ) {
		this( pos, "", "", "" );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Create a <tt>NodeObj</tt> instance with the given parameters.
	 * @param pos the position of the object
	 * @param c the <tt>String</tt> for the center label
	 */
	public FibNodeObj( DPair pos, String c ) {
		this( pos, c, "", "" );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Create a <tt>NodeObj</tt> instance with the given parameters.
	 * @param pos the position of the object
	 * @param c the <tt>String</tt> for the center label
	 * @param l the <tt>String</tt> for the left label
	 * @param r the <tt>String</tt> for the right label
	 */
	public FibNodeObj( DPair pos, String c, String l, String r ) {
		compObj = new CompObj();
		compObj.setPos(pos);

		oval = new OvalObj( new DPair(-NODE_SIZE/2d, -NODE_SIZE/2d),
							new DPair(NODE_SIZE, NODE_SIZE), COLOR);
		oval.setFilled( true );
		oval.setFillColor( Color.white );

		leftLabel = new TextObj(l, new DPair(-NODE_SIZE/2d-SPACE, FONT_SIZE/2d),
								FONT_SIZE, TextObj.RIGHT );
		centerLabel = new TextObj(c, new DPair( 0, FONT_SIZE/2d ),
								FONT_SIZE, TextObj.CENTER );
		rightLabel = new TextObj(r, new DPair(NODE_SIZE/2d+SPACE, FONT_SIZE/2d),
								FONT_SIZE, TextObj.LEFT);
		long recTime = jedas.Jedas.getRecordTime();
		compObj.addItem( oval, jedas.Composite.KEEP_LOW );
		compObj.addItem(leftLabel, jedas.Composite.KEEP_LOW );
		compObj.addItem(centerLabel, jedas.Composite.KEEP_HIGH );
		compObj.addItem(rightLabel, jedas.Composite.KEEP_LOW );

		compObj.setPosTrans( pos );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Lib programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @see NodeItem#getLeftDistance()
	 */
	public double getLeftDistance() {
		return NODE_SIZE/2d+SPACE+leftLabel.getSize().value[0];
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Lib programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @see NodeItem#getRightDistance()
	 */
	public double getRightDistance() {
		return NODE_SIZE/2d+SPACE+rightLabel.getSize().value[0];
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Set the text of the left label.
	 * @param l the <tt>String</tt> for the left label
	 */
	public void setLeftLabel( String l ) {
		leftLabel.setText( l );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @return the text as a <tt>String</tt> of the left label
	 */
	public String getLeftLabel() {
		return leftLabel.getText();
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Set the text of the center label.
	 * @param c the <tt>String</tt> for the center label
	 */
	public void setCenterLabel( String c ) {
		centerLabel.setText( c );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @return the text as a <tt>String</tt> of the center label
	 */
	public String getCenterLabel() {
		return centerLabel.getText();
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Set the text of the left label.
	 * @param r the <tt>String</tt> for the right label
	 */
	public void setRightLabel( String r ) {
		rightLabel.setText( r );
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @return the text as a <tt>String</tt> of the right label
	 */
	public String getRightLabel() {
		return rightLabel.getText();
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Animation programmer</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * Highlight the <tt>NodeObj</tt>. This sets the background color of the
	 * <tt>OvalObj</tt> within this object.
	 * @param b if <i>true</i>, the color is set to light red (<tt>0xffaaaa</tt>),
	 *          otherwise <tt>Color.white</tt>
	 */
	public void setHighlight( boolean b ) {
		if ( b )
			oval.setFillColor(HIGHLIGHT);
		else
			oval.setFillColor(BACKGROUND);
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000> </font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @return alawys <tt>1</tt>
	 * @see NodeItem#getHeight()
	 */
	public int getHeight() {
		return 1;
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Lib internal</font><br>
	 * <b>Since:</b> 0.99rc1, version 0.99rc1<br>
	 * Set the <tt>TreeNode</tt> instance this object is a visualizer for.
	 * The reference is kept in a <tt>transient</tt> variable, and is
	 * thus not serialized on saving.
	 */
	public void setTreeNode( TreeNode treeNode ) {
		this.treeNode = treeNode;
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Lib programmer</font><br>
	 * <b>Since:</b> 0.99rc1, version 0.99rc1<br>
	 * @return the reference to the <tt>TreeNode</tt> this object belongs
	 *         to, or <tt>null</tt> if in player mode or the object does
	 *         not belong to a <tt>TreeNode</tt>.
	 */
	public TreeNode getTreeNode() {
		return treeNode;
	}

	/**
	 * <b>Relevance:</b> <font color=ff0000>Debugging</font><br>
	 * <b>Since:</b> 0.951, version 0.99rc1<br>
	 * @return equivalent to <tt>getName()+" `"+getCenterLabel()+"'"</tt>
	 */
	public String toString() {
		return compObj.getName()+" `"+getCenterLabel()+"'";
	}

	public void setFillColor(Color color) {
		oval.setFillColor(color);
	}
	
	public void setFillColorTrans(Color color) {
		oval.setFillColorTrans(color);
	}

	public Color getFillColor() {
		return oval.getFillColor();
	}

	public void setFilled(boolean b) {
		oval.setFilled(b);
	}

	public void setFilledTrans(boolean b) {
		oval.setFilledTrans(b);
	}

	public boolean isFilled() {
		return oval.isFilled();
	}
}
